<?php

/**
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2022 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

namespace NRFramework\Helpers\Controls;

defined('_JEXEC') or die;

use NRFramework\Helpers\Controls\Control;

class Spacing
{
    /**
     * Returns the CSS of the spacing control.
     * 
     * @param   mixed   $value
     * @param   string  $prefix
     * @param   string  $breakpoint
     * @param   string  $unit
     * 
     * @return  string
     */
    public static function getResponsiveSpacingControlValue($value = '', $prefix = '', $unit = '', $breakpoint = '')
    {
        if (!$value)
        {
            return;
        }

        if ($breakpoint !== '' && !is_string($breakpoint))
        {
            return;
        }
        
        if (is_object($value))
        {
            $value = (array) $value;
        }
        
        // If the value contains units, then use them instead of the given $unit value
        if ($valueUnitData = Control::findUnitinValue($value))
        {
            $value = $valueUnitData['value'];
            $unit = $valueUnitData['unit'];
        }

        $value = self::prepareSpacingControlValue($value, 'desktop');

        if (is_null($value))
        {
            return;
        }

        // Return the value for a specific breakpoint
        if (!empty($breakpoint) && is_string($breakpoint))
        {
            if (!isset($value[$breakpoint]))
            {
                return;
            }

            if (!is_array($value[$breakpoint]) && (string) $value[$breakpoint] !== '0')
            {
                return;
            }

            return $prefix . ': ' . Control::getParsedValue($value[$breakpoint], $unit) . ';';
        }
        
        // Return the value for all breakpoints
        $css = [];

        foreach ($value as $_breakpoint => $values)
        {
            // remove linked property
            if (isset($values['linked']))
            {
                unset($values['linked']);
            }

            $value = Control::getParsedValue($values, $unit);
            if ($value === null || $value === false || $value === '' || (is_array($value) && empty($value)))
            {
                continue;
            }

            $css[$_breakpoint] = $prefix . ': ' . $value . ';';
        }
        
        if (!$css)
        {
            return;
        }
        
        return $css;
    }

    /**
     * Prepares the value
     * 
     * @param   mixed   $value
     * @param   string  $breakpoint
     * 
     * @return  array
     */
    public static function prepareSpacingControlValue($value, $breakpoint = 'desktop')
    {
        if (is_null($value))
        {
            return;
        }

        if (!is_array($value))
        {
            if (empty($breakpoint) || !is_string($breakpoint))
            {
                $breakpoint = 'desktop';
            }
        
            return [
                $breakpoint => $value
            ];
        }

        // If no breakpoint exists in the given value, set it to the given $breakpoint.
        if ((!isset($value['desktop']) && !isset($value['tablet']) && !isset($value['mobile'])) && ($breakpoint && !isset($value[$breakpoint])))
        {
            return [$breakpoint => $value];
        }
        
        return $value;
    }
}